*-----------------------------------------------------------
* Program    : Dumb Terminal v1.6
* Written by : Chuck Kelly
* Date       : Nov-9-2009
* Modified   : Dec-30-2012 Double buffer display fix
* Description: For EASy68K v5.0.0 or later
*  Serial communications terminal program. 
*  Defaults to port Com5, 9600 baud, N, 8, 1, no echo.
*  COMn string contains default com port.
*  F1 - Enter port name
*  F2 - Change baud rate
*  F3 - Echo on/off
*  F4 - Parity
*  F5 - Data bits
*  F6 - Stop bits
*-----------------------------------------------------------

PRINT   MACRO
    ifnc    \1,A1           ; if first parameter is not A1
    lea     \1,A1           ; get string name from macro parameter
    endc
    move.b  #14,d0
    trap    #15             ; display string at (A1) without newline
    ENDM

DOUBLE_BUFFER_ON    MACRO   ; enable double buffering of display
    movem.l d0/d1,-(sp)
    move.b  #92,d0          ;(requires repaint with task #94)
    move.b  #17,d1
    trap    #15
    movem.l (sp)+,d0/d1
    ENDM
    
DOUBLE_BUFFER_OFF   MACRO   ; turn off double buffering
    movem.l d0/d1,-(sp)
    move.b  #92,d0
    move.b  #16,d1
    trap    #15
    movem.l (sp)+,d0/d1
    ENDM

REPAINT MACRO               ; repaint display (necessary when double buffering)
    move.l  d0,-(sp)
    move.b  #94,d0
    trap    #15
    move.l  (sp)+,d0
    ENDM

	ORG	$1000
START:				        ; first instruction of program

    move.b #33,D0
    move.l #1024*$10000+768,d1        Set screen to 1024 x 768
    trap   #15
    
    ; set cursor to bottom left
    move.w  #$001C,d1
    move.b  #11,d0
    trap    #15

    ; check EASy68K version
    move.b  #32,d0
    move.b  #4,d1
    trap    #15             ; version number returned in D1.L
    if.l d1 <lo> #$050000 then.s    ; if older than version 5.0.0
        PRINT   versionError
        bra     halt
    endi

    ; turn off keyboard echo
    move.b  #12,d0
    move.b  #0,d1
    trap    #15

    ; disable simulator shortcut keys
    move.b  #24,d0
    move.l  #1,d1
    trap    #15
    
Init
    ; initialize serial port
    move.l  #1<<16+40,d0    ; PID 1, task 40
    lea     COMn,A1         ; name of port
    trap    #15             ; D0=0 success, 1 invalid PID, 2 error
    
    ; set port parameters
    move.l  #1<<16+41,d0    ; PID 1, task 41
    move.l  portParams,d1   ; port parameters
    trap    #15             ; D0=0 success, 1 invalid PID, 2 error, 3 port not initialized
    if d0 <ne> #0 then.s
        PRINT portError
        clr.b   portOK      ; port not ready
    else
        move.b  #1,portOK   ; port ready
    endi
    
    bsr     mainMenu

loop
    ; check for F1, F2, F3 or F4 keys
    move.l  #$73727170,d1       ; key codes, F4, F3, F2, F1
    move.b  #19,d0
    trap    #15                 ; read key scan codes
    if.b d1 <eq> #$FF then.s    ; if F1 pressed
        bsr     setPort
        bra     Init
    endi
    if.w d1 <eq> #$FF00 then.s  ; if F2 pressed
        bsr     setBaud
        bra     Init
    endi
    if.l d1 <eq> #$00FF0000 then.s  ; if F3 pressed
        if.b F3down <eq> #0 then.s  ; if F3 was up
            if.b echo <eq> #0 then.s    ; if echo is off
                move.b  #1,echo         ; turn echo on
                move.b  #12,d0
                move.b  #1,d1
                trap    #15
            else.s
                move.b  #0,echo         ; turn echo off
                move.b  #12,d0
                move.b  #0,d1
                trap    #15
            endi
            bsr mainMenu
            move.b  #1,F3down   ; remember state of F3 key
        endi
    else.s
        move.b  #0,F3down       ; remember state of F3 key
    endi
    if.l d1 <eq> #$FF000000 then.s  ; if F4 pressed (Parity)
        if.b F4down <eq> #0 then.s  ; if F4 was up
            move.l  portParams,d0
            move.l  d0,d1
            add.l   #PARITY_ADD,d1
            and.l   #PARITY_MASK,d1
            and.l   #~PARITY_MASK,d0
            add.l   d1,d0
            move.l  d0,portParams
            move.b  #1,F4down   ; remember state of F4 key
            bra     Init
        endi
    else.s
        move.b  #0,F4down       ; remember state of F4 key
    endi

    ; check for F5 or F6 keys
    move.l  #$00007574,d1       ; key codes, null, null, F6, F5
    move.b  #19,d0
    trap    #15                 ; read key scan codes
    if.b d1 <eq> #$FF then.s    ; if F5 pressed (nBits)
        if.b F5down <eq> #0 then.s  ; if F5 was up
            move.l  portParams,d0
            move.l  d0,d1
            add.l   #DATA_ADD,d1
            and.l   #DATA_MASK,d1
            if.l d1 <hi> #DATA_MAX then.s
                clr.l d1
            endi
            and.l   #~DATA_MASK,d0
            add.l   d1,d0
            move.l  d0,portParams
            move.b  #1,F5down   ; remember state of F5 key
            bra     Init
        endi
    else.s
        move.b  #0,F5down       ; remember state of F5 key
    endi
    if.w d1 <eq> #$FF00 then.s  ; if F6 pressed (nStops)
        if.b F6down <eq> #0 then.s  ; if F6 was up
            move.l  portParams,d0
            move.l  d0,d1
            not.l   d1
            and.l   #STOP_MASK,d1
            and.l   #~STOP_MASK,d0
            add.l   d1,d0
            move.l  d0,portParams
            move.b  #1,F6down   ; remember state of F6 key
            bra     Init
        endi
    else.s
        move.b  #0,F6down       ; remember state of F6 key
    endi

    if.b portOK <eq> #1 then.l  ; if port ready
        move.l  #7,d0           ; check for keyboard input
        trap    #15             ; d1.b set too 1 if input pending else 0
        if.b d1 <eq> #1 then.s  ; if input pending
            move.b #5,d0        ; get keypress
            trap  #15
          
            ; send keypress
            move.b d1,buf       ; save keypress
            lea    buf,a1
            move.b #1,d1        ; send 1 char
            move.l #1<<16+43,d0 ; PID 1, task 43 send string
            trap   #15          ; D0=0 success, 1 invalid CID, 2 error, 3 port not initialized, 4 timeout
            if.b d0 <eq> #2 or.b d0 <eq> #3 then.s
                bra  Fail
            endi
        endi  
        
        ; check for receive
        lea     buf,a1
        move.b  #15,d1          ; max chars to read
        move.l  #1<<16+42,d0    ; PID 1, task 42 receive string
        trap    #15             ; D0=0 success, 1 invalid CID, 2 error, 3 port not initialized, 4 timeout
        if.b d0 <eq> #2 or.b d0 <eq> #3 then.s
            bra  Fail
        endi
      
        if.b d1 <hi> #0 then.s  ; if characters received
            DOUBLE_BUFFER_ON
            move.b #1,d0        ; display n chars without CR,LR
            lea    buf,a1
            trap   #15
            bsr    mainMenu
            REPAINT
            DOUBLE_BUFFER_OFF
        endi
    endi
    
    ; repaint display (necessary when double buffering)
    move.b  #94,d0
    trap    #15

    bra     loop     

* Communications error
Fail:
    lea     ERROR,A1
    move.b  #13,d0
    trap    #15                 ; display error message

    bra     Init                ; attempt to initialize the port again
    
Halt 
    MOVE.B	#9,D0
	TRAP	#15		            ; halt simulator


* ----------------------------------------
* Display main menu
mainMenu
    movem.l d0-d2,-(SP)

    move.l  #$00800000,d1       ; fill color (dark blue)
    move.b  #81,d0
    trap    #15         

    move.w  #$00FF,d1
    move.b  #11,d0
    trap    #15                 ; get cursor position
    move.w  d1,-(SP)            ; push cursor position
    clr.w   d1
    trap    #15                 ; set cursor to top left
    PRINT   spaces
    move.b  #11,d0
    clr.w   d1
    trap    #15                 ; set cursor to top left
    PRINT   mainMnu
    PRINT   COMn
    move.l  portParams,d1       ; get current speed
    and.l   #BAUD_MASK,d1       ; strip other stuff
    sub.w   #1,d1               ; baud rate number - 1
    mulu.w  #SPEED_WIDTH,d1
    lea     speed,a1
    adda    d1,a1               ; a1 points to speed message
    PRINT   a1
    
    move.l  portParams,d1
    and.l   #PARITY_MASK,d1
    lsr.l   #8,d1
    mulu.w  #FORMAT_WIDTH,d1
    lea     parity,a1
    adda    d1,a1               ; a1 points to parity message
    PRINT   a1
    
    move.l  portParams,d1
    and.l   #DATA_MASK,d1
    move.l  #10,d0
    lsr.l   d0,d1
    mulu.w  #FORMAT_WIDTH,d1
    lea     nBits,a1
    adda    d1,a1               ; a1 points to nBits message
    PRINT   a1
    
    move.l  portParams,d1
    and.l   #STOP_MASK,d1
    move.l  #12,d0
    lsr.l   d0,d1
    mulu.w  #FORMAT_WIDTH,d1
    lea     nStops,a1
    adda    d1,a1               ; a1 points to nStops message
    PRINT   a1
    
    if.b echo <eq> #0 then.s
        lea echoOff,a1
    else.s
        lea echoOn,a1
    endi
    PRINT   a1                  ; display echo on or echo off
    move.w  (SP)+,d1            ; pop cursor position
    move.b  #11,d0
    trap    #15                 ; restore cursor position

    move.l  #$00000000,d1       ; fill color (black)
    move.b  #81,d0
    trap    #15

    movem.l (SP)+,d0-d2
    rts

* ----------------------------------------
* Set the port
setPort
    movem.l d0-d2,-(SP)

    move.l  #DARK_BLUE,d1       ; fill color
    move.b  #81,d0
    trap    #15         

    move.w  #$00FF,d1
    move.b  #11,d0
    trap    #15                 ; get cursor position
    move.w  d1,-(SP)            ; push cursor position
    clr.w   d1
    trap    #15                 ; set cursor to top left
    PRINT   spaces
    move.b  #11,d0
    clr.w   d1
    trap    #15                 ; set cursor to top left
    PRINT   portMnu

    ; turn on keyboard echo
    move.b  #12,d0
    move.b  #1,d1
    trap    #15

    lea     COMn,a1             ; com port name
    move.b  #2,d0
    trap    #15                 ; read string store at (A1)

    move.w  (SP)+,d1            ; pop cursor position
    move.b  #11,d0
    trap    #15                 ; restore cursor position

    move.l  #BLACK,d1           ; fill color
    move.b  #81,d0
    trap    #15

    ; restore keyboard echo
    move.b  #12,d0
    move.b  echo,d1
    trap    #15

    movem.l (SP)+,d0-d2
    rts
        
* ----------------------------------------
* Set baud rate
setBaud
    movem.l d0-d2,-(SP)

    move.l  #DARK_BLUE,d1       ; fill color
    move.b  #81,d0
    trap    #15         

    move.w  #$00FF,d1
    move.b  #11,d0
    trap    #15                 ; get cursor position
    move.w  d1,-(SP)            ; push cursor position

    clr.w   d1
    trap    #15                 ; set cursor to top left
    PRINT   spaces

baudLoop
    move.b  #11,d0
    clr.w   d1
    trap    #15                 ; set cursor to top left

    PRINT   baudMnu
    lea     COMn,a1
    move.b  #14,d0
    trap    #15                 ; display string at (A1) without newline
    move.l  portParams,d1       ; get current speed
    and.l   #BAUD_MASK,d1       ; strip all other stuff
    sub.w   #1,d1               ; baud rate number - 1
    mulu.w  #SPEED_WIDTH,d1
    lea     speed,a1
    adda    d1,a1               ; a1 points to speed message
    PRINT   a1

keyLoop
    move.b  #5,d0               ; get key
    trap    #15
    if.b d1 <eq> #'+' then.s    ; if '+'
        move.l  portParams,d0
        add.b   #1,d0           ; increase speed
        if.b d0 <hi> #BAUD_MAX then.s  ; if max speed
            move.b  #BAUD_MAX,d0   ; no higher
        endi
        move.l  d0,portParams   ; save new speed
        bra baudLoop
    endi
    if.b d1 <eq> #'-' then.s    ; if '-'
        move.l  portParams,d0
        sub.b   #1,d0           ; decrease speed
        if.b d0 <lo> #BAUD_MIN then.s  ; if min speed
            move.b  #BAUD_MIN,d0   ; no lower
        endi
        move.l  d0,portParams   ; save new speed
        bra baudLoop
    endi
    if.b d1 <eq> #CR then.s     ; if 'Enter'
        bra     baudDone        ; exit
    endi
    bra keyLoop    

baudDone
    move.l  #BLACK,d1           ; fill color (black)
    move.b  #81,d0
    trap    #15

    move.w  (SP)+,d1            ; pop cursor position
    move.b  #11,d0
    trap    #15                 ; restore cursor position

    movem.l (SP)+,d0-d2
    rts

* Constants
CR          equ $D
LF          equ $A
DARK_BLUE   equ $00800000
BLACK       equ $00000000

* Serial communications constants
NO_PARITY   equ $00000000
ODD_PARITY  equ $00000100
EVEN_PARITY equ $00000200
MARK_PARITY equ $00000300
PARITY_ADD  equ $00000100
PARITY_MASK equ $00000300
DATA8       equ $00000000
DATA7       equ $00000400
DATA6       equ $00000800
DATA_ADD    equ $00000400
DATA_MAX    equ $00000800
DATA_MASK   equ $00000C00
STOP1       equ $00000000
STOP2       equ $00001000
STOP_MASK   equ $00001000
BAUD96      equ 7
BAUD_MASK   equ $000000FF
BAUD_MAX    equ 14
BAUD_MIN    equ 1

* Variables and Strings
portParams  dc.l    BAUD96+NO_PARITY+DATA8+STOP1  ; 9600 Baud, N, 8, 1
echo    dc.b    0           ; 0= echo off, 1= echo on
COMn    dc.b    'COM4',0    ; default com port
        dcb.b   76,0        ; extra buffer space for user entered com names  
        ds.w    0           ; force word alignment
buf     ds.b    256
cursor  ds.w    1           ; save cursor
portOK  dc.b    0           ; 0= port not ready, 1= port ready
F3down  dc.b    0           ; 0= F3 not pressed, 1= F3 pressed
F4down  dc.b    0           ; 0= F4 not pressed, 1= F4 pressed
F5down  dc.b    0           ; 0= F5 not pressed, 1= F5 pressed
F6down  dc.b    0           ; 0= F6 not pressed, 1= F6 pressed

ERROR           dc.b    CR,LF,'Communications Error',CR,LF,0    
versionError    dc.b    'This program requires EASy68K v5.0.0 or newer.',CR,LF,0
portError       dc.b    'The specified port failed to initialize.',CR,LF,0
mainMnu dc.b    'F1-Port, F2-Baud, F3-Echo, F4-Parity, F5-nBits, F6-nStops  Current: ',0
portMnu dc.b    'Enter port name(e.g. COM5): ',0
baudMnu dc.b    '+ or -   Enter to confirm, baud=',0

SPEED_WIDTH EQU 8           ; width of each speed message
*                *******  *   <-- width of speed message must match SPEED_WIDTH
speed   dc.b    '    110',0
        dc.b    '    300',0
        dc.b    '    600',0
        dc.b    '   1200',0
        dc.b    '   2400',0
        dc.b    '   4800',0
        dc.b    '   9600',0
        dc.b    '  19200',0
        dc.b    '  38400',0
        dc.b    '  56000',0
        dc.b    '  57600',0
        dc.b    ' 115200',0
        dc.b    ' 128000',0
        dc.b    ' 256000',0

FORMAT_WIDTH equ  4         ; width of each format message
*                ***  *   <-- width of each format message must match FORMAT_WIDTH  
parity  dc.b    ', N',0
        dc.b    ', O',0
        dc.b    ', E',0
        dc.b    ', M',0
nBits   dc.b    ', 8',0
        dc.b    ', 7',0
        dc.b    ', 6',0
nStops  dc.b    ', 1',0
        dc.b    ', 2',0

echoOn  dc.b    ', echo on ',0
echoOff dc.b    ', echo off',0
spaces  dc.b    '                                                                                                                                     ',0

	
    END	START		; last line of source
























*~Font name~Courier New~
*~Font size~10~
*~Tab type~1~
*~Tab size~4~
