*************************************************************************************
*														*
*	demo rubber band routines for EASy68K V1.01			2012/04/22		*
*														*
*	when the left mouse is first clicked a rubber band box of a fixed 4:3 aspect	*
*	ratio is started from the mouse x,y position. when the mouse is clicked a	*
*	second time the second corner of the rubber band box is fixed. a third left	*
*	mouse click erases the fixed box and starts a new box.				*
*														*
*	a right mouse click clears any fixed or in progress box being drawn.		*
*														*
*	the box is drawn in EOR mode so it can overlay any graphics without		*
*	obliterating them as it is moved								*
*														*
*	this must be run with exceptions enabled for the mouse interrupt routine.	*
*														*
*	Changes:												*
*														*
*	V1.01 - Changed the main loop timing to use the delay TRAP function. This	*
*		  releases the CPU for the delay time and so reduces the workload.	*
*														*
*	More 68000 and other projects can be found on my website at ..			*
*														*
*	 http://mycorner.no-ip.org/index.html							*
*														*
*	mail : leeedavison@googlemail.com								*
*														*
*************************************************************************************

	ORG		$1000				* set origin

start
	MOVEQ		#12,d0			* keyboard echo
	MOVEQ		#0,d1				* turn off echo
	TRAP		#15

	MOVE.w	#$FF00,d1			* clear screen
	MOVEQ		#11,d0			* position cursor
	TRAP		#15

	BSR		Initialise			* go setup everything

* main loop. everything is done in this loop, the overall speed of the loop is set by
* waiting for a fixed time to pass at the end of the loop

loop
	MOVEQ		#8,d0				* get the time in 1/100 ths seconds
	TRAP		#15

	MOVE.l	d1,-(sp)			* push the time on the stack


	BSR		mouse_buttons		* see if the buttons were clicked
	BSR		mouse_move			* see if the mouse moved


* now wait for a while. this limits the loop to ten redraws a second without which
* this would run way to fast if the machine is capable

	MOVE.l	(sp)+,d7			* restore the main loop start time

	MOVEQ		#8,d0				* get the time in 1/100 ths of seconds
	TRAP		#15

	SUB.l		d1,d7				* subtract the time from the start time
	BLE.s		not_midnight		* if the delta didn't span midnight contimue

	SUB.l		#$83D600,d7			* else subtract a day's worth of 100ths

* moving the wait time into d1 like this means we can have any wait up to 1.27 seconds
* and still use the MOVEQ form to load it

not_midnight
	MOVEQ		#9,d1				* set the wait time in 100ths of a second
	ADD.l		d7,d1				* add the loop -time delta
	BLE		loop				* if the time is up just contimue

	MOVEQ		#23,d0			* else wait d1 100ths of a second
	TRAP		#15

	BRA		loop				* loop forever


*************************************************************************************
*
* draws the rubber band in EOR mode. if both the first and second x,y pairs are unset
* no band is drawn. if only the second x,y pair is unset the band is drawn from the
* first x,y to the mouse x,y. if both x,y pairs are set then the band is drawn from
* the first x,y to the second x,y

rubberband
	MOVEQ		#14,d1			* set EOR colour is drawn
	MOVEQ		#92,d0			* set draw mode
	TRAP		#15

	MOVE.l	#$0000FFFF,d1		* yellow, this could be set from a variable
	MOVEQ		#80,d0			* set pen colour
	TRAP		#15

	MOVE.w	firstY(a3),d2		* get y1
	MOVE.w	firstX(a3),d1		* get x1
	BMI.s		no_draw_band		* skip draw if x1 < 0

	MOVE.w	secondY(a3),d4		* get y2
	MOVE.w	secondX(a3),d3		* get x2
	BPL.s		fixed_band			* if set go draw fixed band

	MOVE.w	lastY(a3),d4		* get y2
	MOVE.w	lastX(a3),d3		* get x2
fixed_band
	MOVEQ		#90,d0			* draw unfilled rectangle from x1,y1 to x2,y2
	TRAP		#15

no_draw_band
	MOVEQ		#4,d1				* set normal draw mode
	MOVEQ		#92,d0			* set draw mode
	TRAP		#15

	RTS


*************************************************************************************
*
* setup stuff

Initialise
	LEA		variables(pc),a3		* variables base address
	BSR		mouse_init			* initialise the mouse

	MOVEQ		#-1,d0			* set for no draw
	MOVE.l	d0,firstX(a3)		* unset the first co-ordinate pair
	MOVE.l	d0,secondX(a3)		* unset the second co-ordinate pair

	MOVE.l	#$028001E0,d1		* set default window size to 640 x 480
	MOVE.l	d1,screen_width(a3)	* save the window size
	MOVEQ 	#33,d0			* get/set window size
	TRAP		#15

	RTS


*************************************************************************************
*
* initialise the mouse routine. set up the interrupt. for the demo to work it must be
* run with exceptions enabled

mouse_init
	LEA		mouse_interrupt(pc),a0	* get the mouse interrupt routine address
	MOVE.l	a0,$64.w			* save the mouse interrupt vector

	MOVE.w	#$0101,d1			* interrupt for mouse down
	MOVEQ		#60,d0			* set mouse interrupt
	TRAP		#15

	MOVEQ		#0,d0				* clear d0
	MOVE.l	d0,left_button(a3)	* clear both button flags

	RTS


*************************************************************************************
*
* mouse interrupt. read the button states and set the button flag if the corresponding
* button is pressed

mouse_interrupt
	MOVEM.l	d0-d1,-(sp)			* save the registers

	MOVEQ		#2,d1				* mouse down state
	MOVEQ		#61,d0			* read the mouse
	TRAP		#15

	MOVEQ		#-1,d1			* set for flag save
	BTST		#0,d0				* was it the left button
	BEQ.s		not_left_button		* if not skip the flag set

	MOVE.w	d1,left_button(a3)	* set the left button flag
not_left_button
	BTST		#1,d0				* was it the right button
	BEQ.s		not_right_button		* if not skip the flag set

	MOVE.w	d1,right_button(a3)	* set the right button flag
not_right_button
	MOVEM.l	(sp)+,d0-d1			* restore the registers
	RTE


*************************************************************************************
*
* check the mouse buttons. on the left button flag if both the first and second x,y
* pairs are unset then the first pair are set. if only the second x,y pair is unset
* then the second pair is set if both x,y pairs are set then the first pair is set
* and the second set are unset starting a new rubber band. on the right button flag
* both pairs are unset

mouse_buttons
	TST.w		left_button(a3)		* test the left button flag
	BEQ.s		test_right_button		* if not set go test the right button flag

	MOVE.w	#0,left_button(a3)	* clear the left button flag

	TST.w		firstX(a3)			* test if we're drawing a rubberband
	BPL.s		start_set			* if start set go test end

	MOVE.l	lastX(a3),firstX(a3)	* save the mouse x,y as the band first x,y
	BRA.s		test_right_button		* go do right button
	
start_set
	TST.w		secondX(a3)			* test if we've drawn a rubberband
	BPL.s		end_set			* if last set go reset

	MOVE.l	lastX(a3),secondX(a3)	* save the mouse x,y as the band second x,y
	BRA.s		test_right_button		* go do right button

end_set
	BSR		rubberband			* erase the rubber band
	MOVE.l	lastX(a3),firstX(a3)	* save the mouse x,y as the band first x,y
	MOVEQ		#-1,d0			* clear the last x,y
	MOVE.l	d0,secondX(a3)		* clear the last co-ordinate pair
	BSR		rubberband			* draw the rubber band
test_right_button
	TST.w		right_button(a3)		* test the right button flag
	BEQ.s		exit_mouse_buttons	* if not set just exit

	BSR		rubberband			* erase the rubber band
	MOVEQ		#-1,d0			* clear x,y
	MOVE.l	d0,firstX(a3)		* clear the first co-ordinate pair
	MOVE.l	d0,secondX(a3)		* clear the last co-ordinate pair

	MOVE.w	#0,right_button(a3)	* clear the right button flag
exit_mouse_buttons
	RTS


*************************************************************************************
*
* see if the mouse moved, if it did possibly redraw the rubber band. if a fixed
* aspect ratio box is not wanted then the call to aspect and the aspect routine
* should be removed

mouse_move
	MOVEQ 	#0,d1				* read mouse state
	MOVEQ 	#61,d0			* read the mouse
	TRAP		#15

	SWAP		d1				* move mouse y to low word
	BSR		aspect			* go do aspect ratio fix

	CMP.l		lastX(a3),d1		* compare old x,y with new x,y
	BEQ.s		mouse_move_exit		* if the same just exit

	MOVE.l	d1,-(sp)			* save new x,y
	BSR		rubberband			* erase the rubber band
	MOVE.l	(sp)+,d1			* restore new x,y

	MOVE.l	d1,lastX(a3)		* save the new x,y
	BSR		rubberband			* draw the rubber band
mouse_move_exit
	RTS


*************************************************************************************
*
* maintain a 4:3 aspect ratio scaled on the relative largest of dx,dy and limited to
* fit into the current window

aspect
	TST.w		firstX(a3)			* is a band started
	BMI		exit_aspect			* if not just exit

	TST.w		secondX(a3)			* is a band done
	BPL		exit_aspect			* if so just exit

* now calculate the absolute dx,dy and flag if they were originally negative

	MOVE.l	d1,d0				* copy x,y
	SWAP		d0				* x to low word

	MOVEQ		#0,d3				* flag both deltas as positive
	SUB.w		firstX(a3),d0		* calculate dx
	BPL.s		dx_not_neg			* branch if not negative

	MOVE.w	#$FF00,d3			* flag negative dx
	NEG.w		d0				* make positive
dx_not_neg
	SUB.w		firstY(a3),d1		* calculate dy
	BPL.s		dy_not_neg			* branch if not negative

	MOVE.b	#$FF,d3			* flag negative dy
	NEG.w		d1				* make positive
dy_not_neg

* now compare 3/4 dx with dy

	MOVE.w	d0,d2				* copy dx
	LSR.w		#1,d2				* / 2
	ADD.w		d0,d2				* make 3/2 x
	LSR.w		#1,d2				* make 3/4 x

	CMP.w		d1,d2				* compare y with 3/4 x
	BCS.s		set_on_y			* if y > 3/4 x go set size on y

* dx was bigger so set dy from dx

	MOVE.w	d2,d1				* set y as 3/4 x
	BRA.s		make_last			* go convert deltas back

* dy was bigger so set dx from dy

set_on_y
	MOVE.w	#$5555,d0			* 1/3 into d2
	MULU.w	d1,d0				* calculate 1/3 y
	ADD.l		#$8000,d0			* round up
	SWAP		d0				* result to low word
	ADD.w		d1,d0				* add y to 1/3 y

* done the aspect ratio fix now convert back to absolute x,y

make_last
	TST.w		d3				* test dx sign flag
	BPL.s		not_neg_dx			* skip negate if not set

	NEG.w		d0				* make dx negative
not_neg_dx
	TST.b		d3				* test dy sign flag
	BPL.s		not_neg_dy			* skip negate if not set

	NEG.w		d1				* make dy negative
not_neg_dy

* got new signed dx,dy now calculate absolute x,y and check if within the current
* window

	ADD.w		firstX(a3),d0		* calculate x
	BPL.s		x_is_positive		* if positive skip low limit

* else x is below the window minimum so fix it and go recalculate on this x

	MOVE.w	firstX(a3),d0		* get dx maximum
	MOVEQ		#1,d1				* make y smallest
	BRA.s		dy_not_neg			* go recalculate

x_is_positive
	CMP.w		xsize(a3),d0		* compare with max + 1
	BCS.s		x_in_range			* if in range skip high limit

* else x is above the window maximum so fix it and go recalculate on this x

	MOVE.w	xsize(a3),d0		* get maximum size + 1
	SUBQ.w	#1,d0				* - 1
	SUB.w		firstX(a3),d0		* calculate dx maximum
	MOVEQ		#1,d1				* make y smallest
	BRA.s		dy_not_neg			* go recalculate

x_in_range
	ADD.w		firstY(a3),d1		* calculate y
	BPL.s		y_is_positive		* if positive skip low limit

* else y is below the window minimum so fix it and go recalculate on this y

	MOVE.w	firstY(a3),d1		* get dy maximum
	MOVEQ		#1,d0				* make x smallest
	BRA.s		dy_not_neg			* go recalculate

y_is_positive
	CMP.w		ysize(a3),d1		* compare with max + 1
	BCS.s		y_in_range			* if in range skip high limit

* else y is above the window maximum so fix it and go recalculate on this y

	MOVE.w	ysize(a3),d1		* get maximum size + 1
	SUBQ.w	#1,d1				* - 1
	SUB.w		firstY(a3),d1		* calculate dy maximum
	MOVEQ		#1,d0				* make x smallest
	BRA.s		dy_not_neg			* go recalculate

* x,y are both in range and dx,dy are fixed at a 4:3 ratio so make the x,y longword

y_in_range
	SWAP		d1				* y to the high word
	MOVE.w	d0,d1				* copy the new x
	SWAP		d1				* x to the high word
exit_aspect
	RTS


*************************************************************************************
*
* variables used

variables						* points to variables base

	OFFSET	0				* going to use relative addressing

left_button
	ds.w	1					* left button flag
right_button
	ds.w	1					* right button flag

firstX
	ds.w	1					* rubberband first x
firstY
	ds.w	1					* rubberband first y
secondX
	ds.w	1					* rubberband second x
secondY
	ds.w	1					* rubberband second y
lastX
	ds.w	1					* last mouse x
lastY
	ds.w	1					* last mouse y

screen_width
xsize	ds.w	1					* window x size
ysize	ds.w	1					* window y size


*************************************************************************************

	END	start


*************************************************************************************
